/*
 * Query the FMS for all outstanding alerts
 */
#include <time.h>
#include <stdio.h>
#include <unistd.h>

#include "libfma.h"
#include "lf_alert.h"
#include "lf_fms.h"

void dump_alerts(int fms, int show_acked, int show_relics);

static void
usage()
{
  fprintf(stderr, "Acknowledge an alert.  This marks an alert as ACKed, possibly causing\n");
  fprintf(stderr, "its deletion.  (See appendix on \"Alerts\" in the README).\n\n");
  fprintf(stderr, "Usage: fm_ack_alert\n");
  fprintf(stderr, "  -h - print this help message\n");
  fprintf(stderr, "  -i <<alert_id> - ACK alert with this ID\n");
  fprintf(stderr, "  -s <fms_server> - specify fms server\n");
  fprintf(stderr, "  -V - print version\n");
  exit(1);
}

int
main(
  int argc,
  char **argv)
{
  int c;
  char *fms_names;
  int fms;
  uint32_t id;
  extern char *optarg;

  /* defaults */
  fms_names = NULL;
  id = 0;

  /* Initialize libfma */
  lf_init();

  /*
   * Get command line args
   */
  while ((c = getopt(argc, argv, "hs:i:V")) != EOF) switch (c) {
  case 'h':
    usage();
    break;
  case 's':
    fms_names = optarg;
    break;
  case 'i':
    id = atoi(optarg);
    break;
  case 'V':
    printf("FMS version is %s\n", Lf_version);
    exit(0);
    break;
  }

  /* make sure we got an alert ID */
  if (id == 0) {
    fprintf(stderr, "Need a valid alert ID\n");
    exit(1);
  }

  /*
   * Open connection to fms
   */
  fms = lf_fms_connect(fms_names);
  if (fms == -1) {
    fprintf(stderr, "Error connecting to ");
    if (fms_names == NULL) {
      fprintf(stderr, "default FMS\n");
    } else {
      fprintf(stderr, "FMS in (%s)\n", fms_names);
    }
    exit(1);
  }

  /*
   * Ack the alert
   */
  lf_ack_alert(fms, id);

  /* all done, shut down connection */
  lf_fms_disconnect(fms);

  exit(0);
}
